/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "mx_auto_config.h"
#include "myriexpress.h"
#include <stdio.h>
#if MX_OS_WINNT
#include "getopt.h"
#include <winsock2.h>
#else
#include <unistd.h>
#include <netinet/in.h>
#endif
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include <assert.h>

#include "mx__lib_types.h"
#include "mcp_logging.h"
#include "mx__lib.h"
#include "mx__driver_interface.h"
#include "mx__fops.h"

#if 0

#define LOG_EVENT_SEND_WIRE(val)					\
((val == 1) ? LOG_EVENT_SEND_WIRE_RAW :					\
 ((val == 2) ? LOG_EVENT_SEND_WIRE_MX :				\
  ((val == 3) ? LOG_EVENT_SEND_WIRE_MX :				\
   ((val == 4) ? LOG_EVENT_SEND_WIRE_MX :				\
    ((val == 5) ? LOG_EVENT_SEND_WIRE_ETHER :				\
     ((val == 6) ? LOG_EVENT_SEND_WIRE_ETHER :				\
      ((val == 7) ? LOG_EVENT_SEND_WIRE_MX :				\
       ((val == 8) ? LOG_EVENT_SEND_WIRE_MX :				\
        ((val == 9) ? LOG_EVENT_SEND_WIRE_MX :			\
	 ((val == 10) ? LOG_EVENT_SEND_WIRE_MX :			\
	  ((val == 11) ? LOG_EVENT_SEND_WIRE_MX :			\
	   ((val == 12) ? LOG_EVENT_SEND_WIRE_MX :			\
	    ((val == 13) ? LOG_EVENT_SEND_WIRE_MX : 0)))))))))))))

#define DMA_CHANNEL_INTR        0
#define DMA_CHANNEL_RECV        1
#define DMA_CHANNEL_SEND        2
#define	DMA_DONE(n) (0x01000000 << (n))

#endif 

typedef struct log_event_t{
  struct log_event_t *next;
  struct log_event_t *prev;
  uint32_t queued;
  uint32_t start;
  uint32_t stop;
  uint32_t category;
  uint32_t length;
  uint32_t val;
} log_event_t;


void
usage()
{
  fprintf(stderr, "Usage: mx_logging [args]\n");
  fprintf(stderr, "-b - Board number [0]\n");
  fprintf(stderr, "-f frequency - Lanai clock rate [313 MHz]\n");
  fprintf(stderr, "-s samples - number of events to retrieve [1024]\n");
  fprintf(stderr, "-h - help\n");
}

int 
main(int argc, char **argv)
{
  mx_endpt_handle_t fd;
  int c;
  extern char *optarg;
  mx_return_t ret;
  mx_get_logging_t get_logging;
  mx_get_logging_strings_t *names;
  mx_get_logging_strings_t tmp;
  mcp_log_t *buffer;
  uint32_t i, board, samples, size, red, blue, green;
  uint32_t cpuc_origin, event, type, data;
  uint32_t colors[125][3], color[5] = {0, 64, 128, 192, 255};
  double clock, time_start, time_stop;

#if 0
  uint32_t j, size, time, type, state;
  uint32_t data0, data1, data2, port, channel, start, stop, queued, category;
  uint32_t isr, length, log_recv_stop, old_isr = 0;
  uint32_t log_send_count[2] = {0, 0}, log_dma_done[4] = {0, 0, 0, 0};
  log_event_t *log_event_ptr, *log_chain_event_ptr; 
  log_event_t *log_dma_head = 0, *log_dma_tail = 0;
  log_event_t *log_lanai_tail = 0;
  log_event_t *log_send_head[2] = {0, 0}, *log_send_tail[2] = {0, 0};
  log_event_t *log_recv_head[2] = {0, 0}, *log_recv_tail[2] = {0, 0};
  log_event_t *log_wire_head[2] = {0, 0}, *log_wire_tail[2] = {0, 0};
  uint16_t old_send_done[2], send_done[2];
  uint16_t old_recv_done[2], recv_done[2];
#endif
  FILE *log;

  board = 0;
  clock = 313.0;
  samples = 1024;
  while ((c = getopt(argc, argv, "b:f:s:h")) != EOF) switch(c) {
  case 'b':
    board = atoi(optarg);
    break;
  case 'f':
    clock = atoi(optarg);
    break;
  case 's':
    samples = atoi(optarg);
    break;
  case 'h':
  default:
    usage();
    exit(1);
  }

  if (samples == 0) {
    fprintf(stderr, "Invalid number of events");
    usage();
    exit(1);
  }

  size = (samples * sizeof (mcp_log_t));
  buffer = (mcp_log_t *) malloc(size);
  if (buffer == 0) {
    perror("buffer allocation failed");
    return 0;
  }
  memset(buffer, 0, size);

  mx_init();
  ret = mx_open_any_board(&fd);
  if (ret != MX_SUCCESS) {
    printf("open failed: %s\n", mx_strerror(ret));
    free(buffer);
    return 0;
  }

  tmp.count = 0;
  if (mx__get_logging_strings(fd, board, &tmp) != 0) {
    perror("get logging count failed");
    free(buffer);
    goto abort_with_fd;
  }
  names = malloc (offsetof(mx_get_logging_strings_t, label) + 128 * tmp.count);
  if (names == NULL) {
    printf("out of memory\n");
    free(buffer);
    goto abort_with_fd;
  }
  names->count = tmp.count;
  if (mx__get_logging_strings(fd, board, names) != 0) {
    perror("get logging strings failed");
    free(names);
    free(buffer);
    goto abort_with_fd;
  }

  get_logging.board_number = board;
  get_logging.size = size;
  get_logging.buffer = (uintptr_t) buffer; 

  if (mx__get_logging(fd, &get_logging) != 0) {
    perror("get logging failed");
    free(names);
    free(buffer);
    goto abort_with_fd;
  }

  log = fopen("log.txt", "w");
  assert(log != 0);
  
  i = 0;
  for (red = 0; red < 5; red++) {
    for (blue = 0; blue < 5; blue++) {
      for (green = 0; green < 5; green++) {
	colors[i][0] = color[red];
	colors[i][1] = color[blue];
	colors[i][2] = color[green];
	i++;
      }
    }
  }
  assert(names->count <= 125);
  for (i = 0; i < names->count; i++) {
    fprintf(log, "Category[ index=%d name=%s topo=State "
	    "color=(%d, %d, %d, 127, true) width=1 < arg=%%d > ]\n",
	    i, names->label[i], colors[i][0], colors[i][1], colors[i][2]);
  }
  
#if 0
  fprintf(log, "Category[ index=%d name=Send_Wire_Raw topo=State "
	  "color=(255,255,0,127,true) width=1 < type=%%d, length=%%d > ]\n",
	  LOG_EVENT_SEND_WIRE_RAW);
  fprintf(log, "Category[ index=%d name=Send_Wire_MX topo=State "
	  "color=(0,255,255,127,true) width=1 < type=%%d, length=%%d > ]\n", 
	  LOG_EVENT_SEND_WIRE_MX);
  fprintf(log, "Category[ index=%d name=Send_Wire_Ether topo=State "
	  "color=(127,127,127,127,true) width=1 < type=%%d, length=%%d > ]\n", 
	  LOG_EVENT_SEND_WIRE_ETHER);
  fprintf(log, "Category[ index=%d name=Recv_Wire_Raw topo=State "
	  "color=(255,255,0,127,true) width=1 < length=%%d > ]\n",
	  LOG_EVENT_RECV_WIRE_RAW);
  fprintf(log, "Category[ index=%d name=Recv_Wire_MX topo=State "
	  "color=(0,255,255,127,true) width=1 < length=%%d > ]\n", 
	  LOG_EVENT_RECV_WIRE_MX);
  fprintf(log, "Category[ index=%d name=Recv_Wire_Ether topo=State "
	  "color=(127,127,127,127,true) width=1 < length=%%d > ]\n", 
	  LOG_EVENT_RECV_WIRE_ETHER);
  fprintf(log, "Category[ index=%d name=Send_FIFO topo=Arrow "
	  "color=(127,255,255,127,true) width=1 ]\n", LOG_EVENT_SEND_FIFO);
  fprintf(log, "Category[ index=%d name=Recv_FIFO topo=Arrow "
	  "color=(255,127,255,127,true) width=1 ]\n", LOG_EVENT_RECV_FIFO);
  fprintf(log, "Category[ index=%d name=DMA_FIFO topo=Arrow "
	  "color=(255,255,127,127,true) width=1 ]\n", LOG_EVENT_DMA_FIFO);
  
  for (port = 0; port < 2; port++) {
    old_send_done[port] = ntohs(buffer[0].send_done[port]);
    old_recv_done[port] = ntohs(buffer[0].recv_done[port]);
  }
  
  for (i = 0; i < samples; i++) {
    time = ntohl(buffer[i].cpuc) - ntohl(buffer[0].cpuc);
    isr = ntohl(buffer[i].isr);
    type = buffer[i].type;
    state = buffer[i].state;
    data0 = ntohs(buffer[i].data0);
    data1 = ntohs(buffer[i].data1);
    data2 = ntohl(buffer[i].data2);
    
    for (channel = 0; channel < 4; channel++) {
      if (((isr & DMA_DONE(channel)) == 0) 
	  && ((old_isr & DMA_DONE(channel)) != 0)) {
	//printf("[%d]: dma_done CLEAR: channel %d, time %d\n", 
	//       i, channel, time);
	log_dma_done[channel] = 0;
      }
      if (((isr & DMA_DONE(channel)) != 0) 
	  && ((old_isr & DMA_DONE(channel)) == 0)) {
	//printf("[%d]: dma_done: channel %d, time %d\n", i, channel, time);
	log_dma_done[channel] = time;
      }
    }
    old_isr = isr;

    for (port = 0; port < 2; port++) {
      send_done[port] = buffer[i].send_done[port];
      recv_done[port] = buffer[i].recv_done[port];
      
      if (send_done[port] > old_send_done[port]) {
	//printf("[%d]: send done: old %d, new %d\n", 
	//     i, old_send done[port], send_done[port]);
	for (j = old_send_done[port]; j < send_done[port]; j++) {
	  log_event_ptr = log_send_head[port];
	  while (log_event_ptr != 0) {
	    //printf("[%d]: send done: time %d, j %d, event 0x%p, val %d\n", 
	    // i, time, j, log_event_ptr, log_event_ptr->val);
	    if (log_event_ptr->val == j) {
	      assert(log_event_ptr->stop == 0);
	      log_event_ptr->stop = time;
	      break;
	    }
	    log_event_ptr = log_event_ptr->next;
	  }
	}
      }
      old_send_done[port] = send_done[port];

      if (recv_done[port] > old_recv_done[port]) {
	//printf("[%d]: recv done: old %d, new %d\n", i, 
	//  old_recv_done[port], recv_done[port]);
	for (j = old_recv_done[port]; j < recv_done[port]; j++) {
	  log_event_ptr = malloc(sizeof (log_event_t));
	  assert(log_event_ptr != 0);
	  log_event_ptr->next = 0;
	  log_event_ptr->stop = time;
	  log_event_ptr->val = j;
	  
	  //printf("[%d]: recv done: time %d, j %d, event 0x%p\n", 
	  // i, time, j, log_event_ptr);
	  
	  if (log_recv_tail[port] == 0) {
	    assert(log_recv_head[port] == 0);
	    log_recv_head[port] = log_event_ptr;
	    log_recv_tail[port] = log_event_ptr;
	  } else {
	    assert(log_recv_head[port] != 0);
	    log_recv_tail[port]->next = log_event_ptr;
	    log_recv_tail[port] = log_event_ptr;
	  }
	}
	old_recv_done[port] = recv_done[port];
      }
    }

    switch (type) {
    case LOG_TYPE_LANAI:
      
      switch (state) {
      case LOG_STATE_START:
	log_event_ptr =  malloc(sizeof (log_event_t));
	assert(log_event_ptr != 0);
	log_event_ptr->next = 0;
	log_event_ptr->prev = log_lanai_tail;
	log_event_ptr->queued = time;
	log_event_ptr->category = data0;
	log_event_ptr->prev = log_lanai_tail;
	log_lanai_tail = log_event_ptr;
	break;

      case LOG_STATE_STOP:
	log_event_ptr = log_lanai_tail;
	if (log_event_ptr != 0) {
	  assert(log_event_ptr->category == data0);
	
	  fprintf(log, "Primitive[ TimeBBox(%f,%f) Category=%d (%f, %d) "
		  "(%f, %d) < arg=%d > ]\n", 
		  log_event_ptr->queued/clock, time/clock, data0, 
		  log_event_ptr->queued/clock, type, time/clock, type, data1);
	  
	  log_lanai_tail = log_event_ptr->prev;
	  free(log_event_ptr);
	}
	break;

      default:
	printf("Unknown log lanai state (%d)\n", state);
	exit(1);
      }
      break;
      
    case LOG_TYPE_PCI:
      
      switch (state) {
      case LOG_STATE_START:
	log_event_ptr = malloc(sizeof (log_event_t));
	assert(log_event_ptr != 0);
	log_event_ptr->next = 0;
	log_event_ptr->prev = log_dma_tail;
	log_event_ptr->queued = time;
	log_event_ptr->category = data0;
	log_event_ptr->length = data1;
	log_event_ptr->val = data2;
	assert(data2 != 0);
	
	//printf("[%d]: dma_start: event 0x%p, time %d, val %d\n", 
	//     i, log_event_ptr, time, data2);
	
	if (log_dma_tail == 0) {
	  assert(log_dma_head == 0);
	  log_dma_head = log_event_ptr;
	  log_dma_tail = log_event_ptr;
	} else {
	  assert(log_dma_head != 0);
	  
	  /* if we find a previous start event with the same pcidma, 
	     that means it's a chained DMA */
	  log_chain_event_ptr = log_dma_head;
	  do {
	    if (log_chain_event_ptr->val == data2) {
	      log_chain_event_ptr->length += data1;
	      free(log_event_ptr);
	      break;
	    }
	    log_chain_event_ptr = log_chain_event_ptr->next;
	  } while (log_chain_event_ptr != 0);
	  
	  /* no chained DMA */
	  if (log_chain_event_ptr == 0) {
	    log_dma_tail->next = log_event_ptr;
	    log_dma_tail = log_event_ptr;
	  }
	}
	break;
	
      case LOG_STATE_STOP:
	log_event_ptr = log_dma_head;
	while (log_event_ptr != 0) {
	  if (log_event_ptr->val == data2) {
	    channel = data0;
	    start = log_event_ptr->start;
	    queued = log_event_ptr->queued;
	    category = log_event_ptr->category;
	    length = log_event_ptr->length;
	    
	    //printf("[%d]: dma_stop: event 0x%p, queued %d, time %d, val %d, "
	    //   "len %d, channel %d, dma_count %d\n", i, log_event_ptr, 
	    //   queued, time, data2, length, data0, data1);
	    
	    assert(log_dma_done[channel] != 0);
	    stop = log_dma_done[channel];
	    log_dma_done[channel] = time;
	    
	    start = stop - (length*clock / ((length > 1024) ? 1000.0 : 500.0));
	    if (start < queued) {
	      start = queued + 1;
	    }
	    
	    log_dma_head = log_event_ptr->next;
	    if (log_event_ptr->next == 0) {
	      log_dma_tail = 0;
	    }
	    free(log_event_ptr);
	    
	    fprintf(log, "Primitive[ TimeBBox(%f,%f) Category=%d (%f, %d) "
		    "(%f, %d) ]\n", queued/clock, start/clock, 
		    LOG_EVENT_DMA_FIFO, queued/clock, LOG_TYPE_LANAI, 
		    start/clock, type);
	    fprintf(log, "Primitive[ TimeBBox(%f,%f) Category=%d (%f, %d) "
		    "(%f, %d) < arg=%d > ]\n", 
		    start/clock, stop/clock, category, start/clock, type, 
		    stop/clock, type, length);
	    fprintf(log, "Primitive[ TimeBBox(%f,%f) Category=%d (%f, %d) "
		    "(%f, %d) ]\n", stop/clock, time/clock, 
		    LOG_EVENT_DMA_FIFO, stop/clock, type, 
		    time/clock, LOG_TYPE_LANAI);
	    break;
	  }
	  
	  log_event_ptr = log_event_ptr->next;
	}
	break;

      default:
	printf("Unknown log dma state (%d)\n", state);
	exit(1);
      }
      break;
      
    case LOG_TYPE_SEND_P0:
    case LOG_TYPE_SEND_P1:

      port = type - LOG_TYPE_SEND_P0;
      switch (state) {
      case LOG_STATE_START:
	log_event_ptr = malloc(sizeof (log_event_t));
	assert(log_event_ptr != 0);
	log_event_ptr->next = 0;
	log_event_ptr->queued = time;
	log_event_ptr->stop = 0;
	log_event_ptr->category = data0;
	log_event_ptr->length = data1;
	log_event_ptr->val = data2;

	/* check on pkt type, sync with LOG_EVENT_SEND_WIRE */
	assert(data2 < 14);
	
	log_send_count[port]++;
	//printf("[%d]: send_start: time %d, cat %d, val %d, event 0x%p\n", 
	//     i, time, data0, data2, log_event_ptr);

	if (log_send_tail[port] == 0) {
	  assert(log_send_head[port] == 0);
	  log_send_head[port] = log_event_ptr;
	  log_send_tail[port] = log_event_ptr;
	} else {
	  assert(log_send_head[port] != 0);
	  log_send_tail[port]->next = log_event_ptr;
	  log_send_tail[port] = log_event_ptr;
	}
	break;
	
      case LOG_STATE_STOP:
	//printf("[%d]: send_stop : count %d, data2 %d, head 0x%p\n", 
	//i, log_send_count[port], data2, log_send_head[port]);
	log_event_ptr = log_send_head[port];
	if (log_send_count[port] == data2) {
	  assert(log_event_ptr->stop != 0);
	  assert(log_event_ptr->category == data0);
	  queued = log_event_ptr->queued;
	  stop = log_event_ptr->stop;
	  length = log_event_ptr->length;
	  start = stop - (length * clock / 250.0);

	  log_send_head[port] = log_event_ptr->next;
	  if (log_event_ptr->next == 0) {
	    log_send_tail[port] = 0;
	  }
	  free(log_event_ptr);
	  
	  fprintf(log, "Primitive[ TimeBBox(%f,%f) Category=%d (%f, %d) "
		  "(%f, %d) ]\n", queued/clock, start/clock, 
		  LOG_EVENT_SEND_FIFO, queued/clock, LOG_TYPE_LANAI, 
		  start/clock, type);
	  fprintf(log, "Primitive[ TimeBBox(%f,%f) Category=%d (%f, %d) "
		  "(%f, %d) < type=%d, length=%d > ]\n", 
		  start/clock, stop/clock, LOG_EVENT_SEND_WIRE(data0), 
		  start/clock, type, stop/clock, type, data0, length);
	  fprintf(log, "Primitive[ TimeBBox(%f,%f) Category=%d (%f, %d) "
		  "(%f, %d) ]\n", stop/clock, time/clock, 
		  LOG_EVENT_SEND_FIFO, stop/clock, type, 
		  time/clock, LOG_TYPE_LANAI);
	  
	  log_send_count[port]--;
	}
	
	log_event_ptr = log_send_head[port];
	while (log_event_ptr != 0) {
	  log_event_ptr->val--;
	  log_event_ptr = log_event_ptr->next;
	}
	break;
	
      default:
	printf("Unknown log send state (%d)\n", state);
	exit(1);
      }
      break;
      
    case LOG_TYPE_RECV_P0:
    case LOG_TYPE_RECV_P1:
      port = type - LOG_TYPE_RECV_P0;
      channel = data1;
      log_event_ptr = log_recv_head[port];
      assert(state == LOG_STATE_STOP);
      assert(recv_done[port] > 0);

      //printf("[%d]: recv stop : channel %d, head 0x%p, val %d\n", 
      //    i, channel, log_event_ptr,
      //   ((log_event_ptr != 0) ? log_event_ptr->val : 999));
      if ((log_event_ptr != 0) && (log_event_ptr->val == 0)) {
	
	stop = log_event_ptr->stop;
	log_event_ptr->queued = time;
	log_event_ptr->category = data0;
	log_event_ptr->length = data2;
	
	//printf("[%d]: -> len %d, time %d\n", i, length, time);
  
	log_recv_head[port] = log_event_ptr->next;
	if (log_event_ptr->next == 0) {
	  log_recv_tail[port] = 0;
	}
	
	log_event_ptr->next = 0;
	if (log_wire_tail[port] == 0) {
	  assert(log_wire_head[port] == 0);
	  log_event_ptr->prev = 0;
	  log_wire_head[port] = log_event_ptr;
	  log_wire_tail[port] = log_event_ptr;
	} else {
	  assert(log_recv_head[port] != 0);
	  log_event_ptr->prev = log_wire_tail[port];
	  log_wire_tail[port]->next = log_event_ptr;
	  log_wire_tail[port] = log_event_ptr;
	}
      }

      log_event_ptr = log_recv_head[port];
      while (log_event_ptr != 0) {
	log_event_ptr->val--;
	log_event_ptr = log_event_ptr->next;
      }
      break;
      
    default:
      printf("Unknown log type (%d)\n", type);
      exit(1);
    }
  }

  for (port = 0; port < 2; port++) {
    log_recv_stop = -1;
    log_event_ptr = log_wire_tail[port];
    while (log_event_ptr != 0) {
      stop = log_event_ptr->stop;
      length = log_event_ptr->length;
      
      //printf("recv rewind 1: stop %d, recv_stop %d, length %d\n", 
      //   stop, log_recv_stop, length);

      if (stop > log_recv_stop) {
	stop = log_recv_stop;
	log_event_ptr->stop = stop;
      }
      
      if (stop > (length * clock / 250.0)) {
	start = stop - (length * clock / 250.0);
      } else {
	log_event_ptr->prev = 0;
	log_wire_head[port] = log_event_ptr;
	start = 0;
      }

      //printf("---> start %d, stop %d, recv_stop %d\n", start, stop, start);

      log_recv_stop = start;
      log_event_ptr->start = start;
      log_event_ptr = log_event_ptr->prev;
    }
  }
	
  for (port = 0; port < 2; port++) {
    type = LOG_TYPE_RECV_P0 + port;
    log_event_ptr = log_wire_head[port];
    while (log_event_ptr != 0) {
      start = log_event_ptr->start;
      stop = log_event_ptr->stop;
      time = log_event_ptr->queued;
      length = log_event_ptr->length;
      category = log_event_ptr->category;
      log_event_ptr = log_event_ptr->next;
      
      //printf("recv rewind 2: start %d, stop %d, time %d, length %d\n", 
      //    start, stop, time, length);

      assert(time > stop);
      assert(stop > start);
      fprintf(log, "Primitive[ TimeBBox(%f,%f) Category=%d (%f, %d) "
	      "(%f, %d) < length=%d > ]\n", 
	      start/clock, stop/clock, category, 
	      start/clock, type, stop/clock, type, length);
      fprintf(log, "Primitive[ TimeBBox(%f,%f) Category=%d (%f, %d) "
	      "(%f, %d) ]\n", stop/clock, time/clock, 
	      LOG_EVENT_RECV_FIFO, stop/clock, type, 
	      time/clock, LOG_TYPE_LANAI);
    }
  }

#endif

  cpuc_origin = ntohl(buffer[0].cpuc_start);
  
  for (i = 0; i < samples; i++) {
    time_start = (ntohl(buffer[i].cpuc_start) - cpuc_origin) / clock;
    time_stop = (ntohl(buffer[i].cpuc_stop) - cpuc_origin) / clock;
    event = buffer[i].event;
    type = LOG_TYPE_LANAI;
    data = 0;
    
    fprintf(log, "Primitive[ TimeBBox(%f,%f) Category=%d (%f, %d) "
	    "(%f, %d) < arg=%d > ]\n", time_start, time_stop, event, 
	    time_start, type, time_stop, type, data);
  }

  free(names);
  free(buffer);

 abort_with_fd:
  mx__close(fd);
  mx_finalize();

  return 0;
}
